# WinVM-IOPS-Check.ps1 (v1.7)
# Checks Windows VM settings for low-IOPS template readiness
# Updates:
#  - Fix recommendations collection (no [ref] += issues)
#  - CIRCLE service names: circle-agent, circle-watchdog
#  - CIRCLE notify autostart via Task Scheduler (no Startup shortcut)
#  - Simplified user/hive assumptions: run by cloud user in elevated PS; use HKCU only (plus HKU\.DEFAULT)
# RunCmd:
#  powershell -ExecutionPolicy Bypass -File .\WinVM-IOPS-Check.ps1
# ASCII-only comments

# ASCII-only comments

param(
  # Expected fixed pagefile size (MB). Default 4096 (4G).
  [int]$ExpectedPagefileMB = 4096,

  # Expected EventLog max sizes (MB)
  [int]$ExpectedAppLogMB = 4,
  [int]$ExpectedSysLogMB = 4,
  [int]$ExpectedSecLogMB = 20
)

$ErrorActionPreference = "Continue"

function Say($msg="") { Write-Host $msg }

function ToGB([double]$bytes) {
  [Math]::Round($bytes / 1GB, 2)
}

# Recommendations bucket (ArrayList: robust Add)
$recs = New-Object System.Collections.ArrayList

function Add-Rec([string]$level, [string]$msg) {
  [void]$recs.Add(("  {0}: {1}" -f $level, $msg))
}

function Get-ServiceStartType($name) {
  try {
    $svc = Get-CimInstance Win32_Service -Filter ("Name='{0}'" -f $name) -ErrorAction Stop
    return $svc.StartMode
  } catch {
    return "Unknown"
  }
}

function Format-ScanDay($day) {
  switch ($day) {
    0 { "Everyday" }
    1 { "Sunday" }
    2 { "Monday" }
    3 { "Tuesday" }
    4 { "Wednesday" }
    5 { "Thursday" }
    6 { "Friday" }
    7 { "Saturday" }
    8 { "Never" }
    default { "$day" }
  }
}

# Expectations helper (services)
function Expect-ServiceInline($svcState, [string]$name, $wantStartTypes, $wantStatus = $null) {
  if (-not $svcState.ContainsKey($name)) {
    Add-Rec "WARN" ("Service {0} not checked." -f $name)
    return
  }
  $st = $svcState[$name]
  if ($wantStartTypes -isnot [System.Array]) { $wantStartTypes = @($wantStartTypes) }

  if ($wantStartTypes -notcontains $st.StartType) {
    Add-Rec "WARN" ("Service {0} StartType is {1} (expected one of: {2})." -f $name, $st.StartType, ($wantStartTypes -join ", "))
  }
  if ($wantStatus -and ($st.Status -ne $wantStatus)) {
    Add-Rec "WARN" ("Service {0} Status is {1} (expected {2})." -f $name, $st.Status, $wantStatus)
  }
}

# Small env getter (no duplicates)
function Get-Env([string]$name, [string]$scope) {
  try { return [Environment]::GetEnvironmentVariable($name, $scope) } catch { return $null }
}

function Get-RegStr($path, $name) {
  try { (Get-ItemProperty $path -Name $name -ErrorAction Stop).$name } catch { $null }
}

# Parse sc.exe qfailure output (best effort)
function Get-ScFailureSummary([string]$svcName) {
  try {
    $lines = & sc.exe qfailure $svcName 2>$null
    if (-not $lines) { return $null }
    $txt = ($lines -join "`n")

    $reset = $null
    if ($txt -match 'RESET_PERIOD.*:\s*(\d+)') { $reset = [int]$Matches[1] }

    $hasRestart = ($txt -match 'RESTART')
    return @{
      Text       = $txt
      Reset      = $reset
      HasRestart = $hasRestart
    }
  } catch {
    return $null
  }
}

# Compare recovery policy to expected: reset=86400, restart/180000 x3, failureflag=1
function Validate-CircleServiceRecovery([string]$svcName, [string]$displayName) {
  $sum = Get-ScFailureSummary $svcName
  if (-not $sum) {
    Add-Rec "INFO" ("{0} recovery info unavailable (sc qfailure failed)." -f $displayName)
    return
  }

  if (-not $sum.HasRestart) {
    Add-Rec "WARN" ("{0} recovery actions do not include RESTART." -f $displayName)
  }

  if ($sum.Reset -ne $null -and $sum.Reset -ne 86400) {
    Add-Rec "WARN" ("{0} recovery reset period is {1}s (expected 86400)." -f $displayName, $sum.Reset)
  } elseif ($sum.Reset -eq $null) {
    Add-Rec "INFO" ("{0} recovery reset period not parsed (verify via sc qfailure)." -f $displayName)
  }

  # Validate restart delay(s) 180000 (best effort)
  if ($sum.Text -notmatch '180000') {
    Add-Rec "WARN" ("{0} recovery does not mention 180000ms delay (expected restart/180000 x3)." -f $displayName)
  }

  # failureflag (non-crash) check (best effort parse)
  try {
    $ff = & sc.exe qfailureflag $svcName 2>$null
    if ($ff) {
      $t = ($ff -join "`n")
	  if ($t -match 'FAILURE_ACTIONS_ON_NONCRASH_FAILURES\s*:\s*(TRUE|FALSE|[01])') {
		$raw = $Matches[1].ToUpperInvariant()
		$ok = ($raw -eq "TRUE" -or $raw -eq "1")
		if (-not $ok) {
			Add-Rec "WARN" ("{0} failureflag is {1} (expected TRUE/1)." -f $displayName, $Matches[1])
		}
	  } else {
		Add-Rec "INFO" ("{0} failureflag output not parsed; verify manually if needed." -f $displayName)
	  }
    } else {
      Add-Rec "INFO" ("{0} failureflag info unavailable (sc qfailureflag empty)." -f $displayName)
    }
  } catch {
    Add-Rec "INFO" ("{0} failureflag query failed." -f $displayName)
  }
}

Say "=== Windows VM low-IOPS template check (v1.7) ==="
Say ("Computer: {0}" -f $env:COMPUTERNAME)
Say ("User: {0}" -f $env:USERNAME)

# One-time queries
$os = $null
$cs = $null
try { $os = Get-CimInstance Win32_OperatingSystem } catch {}
try { $cs = Get-CimInstance Win32_ComputerSystem } catch {}

if ($os -and $os.Caption) { Say ("OS: {0}" -f $os.Caption) }
else { Say "OS: (unknown)" }
Say ""

# -----------------------
# Memory / Commit (uses $os/$cs once)
# -----------------------
Say "Memory:"
try {
  if ($os -and $cs) {
    $commitUsedGB  = ToGB([double]$os.TotalVirtualMemorySize * 1KB - [double]$os.FreeVirtualMemory * 1KB)
    $commitLimitGB = ToGB([double]$os.TotalVirtualMemorySize * 1KB)
    $ramGB         = [Math]::Round(([double]$cs.TotalPhysicalMemory / 1GB), 2)

    Say ("  RAM (GB):             {0}" -f $ramGB)
    Say ("  Commit used (GB):     {0}" -f $commitUsedGB)
    Say ("  Commit limit (GB):    {0}" -f $commitLimitGB)
  } else {
    Say "  (unable to query)"
  }
} catch {
  Say "  (unable to query)"
}
Say ""

# -----------------------
# Pagefile
# -----------------------
Say "Pagefile:"
$pfReg = $null
try {
  $pfReg = (Get-ItemProperty "HKLM:\SYSTEM\CurrentControlSet\Control\Session Manager\Memory Management" -ErrorAction Stop).PagingFiles
  Say ("  Registry PagingFiles: {0}" -f ($pfReg -join " | "))
} catch {
  Say "  Registry PagingFiles: (unable to read)"
  Add-Rec "WARN" "Registry PagingFiles could not be read."
}

try {
  if ($cs -and ($cs.AutomaticManagedPagefile -ne $false)) {
    Add-Rec "WARN" "AutomaticManagedPagefile is not False (pagefile may be auto-managed)."
  }
} catch {}

try {
  if ($pfReg) {
    $want = $ExpectedPagefileMB
    $hit = $false
    foreach ($line in $pfReg) {
      if ($line -match '^[Cc]:\\pagefile\.sys\s+(\d+)\s+(\d+)$') {
        $i = [int]$Matches[1]; $m = [int]$Matches[2]
        if ($i -eq $want -and $m -eq $want) {
          $hit = $true
        } else {
          Add-Rec "WARN" ("Registry PagingFiles is '{0}' (expected {1}/{1} MB)." -f $line, $want)
        }
      }
    }
    if (-not $hit) {
      Add-Rec "WARN" ("Registry PagingFiles does not contain C:\pagefile.sys {0} {0}." -f $ExpectedPagefileMB)
    }
  }
} catch {}

$pfs = $null
try { $pfs = Get-CimInstance Win32_PageFileSetting -ErrorAction SilentlyContinue } catch {}
if ($pfs) {
  foreach ($p in $pfs) {
    Say ("  {0}  InitialMB={1}  MaximumMB={2}" -f $p.Name, $p.InitialSize, $p.MaximumSize)
    if ($p.Name -match 'pagefile\.sys') {
      if (($p.InitialSize -ne $ExpectedPagefileMB) -or ($p.MaximumSize -ne $ExpectedPagefileMB)) {
        Add-Rec "WARN" ("WMI PageFileSetting {0} is {1}/{2} MB (expected {3}/{3} MB)." -f $p.Name, $p.InitialSize, $p.MaximumSize, $ExpectedPagefileMB)
      }
    }
  }
} else {
  Say "  Win32_PageFileSetting: (none)"
  Add-Rec "WARN" "Win32_PageFileSetting is empty/unavailable (cannot confirm fixed pagefile via WMI)."
}

$pfu = $null
try { $pfu = Get-CimInstance Win32_PageFileUsage -ErrorAction SilentlyContinue } catch {}
if ($pfu) {
  foreach ($u in $pfu) {
    Say ("  Usage: {0}  AllocatedMB={1}  CurrentUsedMB={2}  PeakUsedMB={3}" -f $u.Name, $u.AllocatedBaseSize, $u.CurrentUsage, $u.PeakUsage)
  }
} else {
  Say "  Win32_PageFileUsage: (none)"
}
Say ""

# -----------------------
# Disk (basic)
# -----------------------
try {
  $disk = Get-CimInstance Win32_DiskDrive | Select-Object -First 1
  if ($disk) {
    $sizeGB = [Math]::Round($disk.Size / 1GB, 0)
    Say "Disk (basic):"
    Say ("  Model={0}  Interface={1}  SizeGB={2}" -f $disk.Model, $disk.InterfaceType, $sizeGB)
    Say ""
  }
} catch {}

# -----------------------
# TEMP/TMP
# -----------------------
Say "TEMP/TMP (environment):"

$procTEMP = Get-Env "TEMP" "Process"
$procTMP  = Get-Env "TMP"  "Process"
$userTEMP = Get-Env "TEMP" "User"
$userTMP  = Get-Env "TMP"  "User"

Say ("  Process TEMP={0}" -f $(if ($procTEMP) { $procTEMP } else { "(missing)" }))
Say ("  Process TMP ={0}" -f $(if ($procTMP)  { $procTMP  } else { "(missing)" }))
Say ("  User    TEMP={0}" -f $(if ($userTEMP) { $userTEMP } else { "(missing)" }))
Say ("  User    TMP ={0}" -f $(if ($userTMP)  { $userTMP  } else { "(missing)" }))

$wantTemp = Join-Path $env:USERPROFILE "AppData\Local\Temp"
Say ("  Expected (baseline) ~ {0}" -f $wantTemp)

if (-not $userTEMP) {
  Add-Rec "WARN" "User-level TEMP is missing (HKCU\\Environment)."
} elseif ($userTEMP -ne $wantTemp) {
  Add-Rec "INFO" ("User-level TEMP is '{0}' (expected '{1}')." -f $userTEMP, $wantTemp)
}
if (-not $userTMP) {
  Add-Rec "WARN" "User-level TMP is missing (HKCU\\Environment)."
} elseif ($userTMP -ne $wantTemp) {
  Add-Rec "INFO" ("User-level TMP is '{0}' (expected '{1}')." -f $userTMP, $wantTemp)
}

Say ""

# -----------------------
# Defender
# -----------------------
Say "Defender:"
try {
  $st = Get-MpComputerStatus
  Say ("  AMServiceEnabled:           {0}" -f $st.AMServiceEnabled)
  Say ("  RealTimeProtectionEnabled:  {0}" -f $st.RealTimeProtectionEnabled)
  Say ("  OnAccessProtectionEnabled:  {0}" -f $st.OnAccessProtectionEnabled)
  Say ("  AntivirusEnabled:           {0}" -f $st.AntivirusEnabled)
  Say ("  NISEnabled:                 {0}" -f $st.NISEnabled)

  $mp = Get-MpPreference
  $day = $mp.ScanScheduleDay
  Say ("  ScanScheduleDay:            {0}" -f (Format-ScanDay $day))

  try {
    $t = $mp.ScanScheduleTime
    if ($t) { Say ("  ScanScheduleTime:           {0}" -f ($t.ToString("HH:mm:ss"))) }
    else    { Say "  ScanScheduleTime:           (not set)" }
  } catch {
    Say "  ScanScheduleTime:           (unavailable)"
  }

  $ex = @($mp.ExclusionPath)
  Say ("  ExclusionPath count:        {0}" -f $ex.Count)
  $show = $ex | Select-Object -First 10
  if ($show.Count -gt 0) {
    Say "  ExclusionPath (first 10):"
    foreach ($p in $show) { Say ("    {0}" -f $p) }
  }

  if ($mp.ScanScheduleDay -ne 8) {
    Add-Rec "WARN" ("Defender ScanScheduleDay is {0} (expected Never)." -f (Format-ScanDay $mp.ScanScheduleDay))
  }

  $expectedEx = @(
    "C:\Windows\Temp",
    "C:\Windows\SoftwareDistribution",
    "C:\ProgramData\Microsoft\Windows Defender",
    "C:\pagefile.sys"
  )
  foreach ($e in $expectedEx) {
    if ($ex -notcontains $e) { Add-Rec "WARN" ("Defender exclusion missing: {0}" -f $e) }
  }

} catch {
  Say "  WARN: Defender cmdlets not available or access denied."
  Add-Rec "WARN" "Defender preferences unavailable (cannot validate schedule/exclusions)."
}
Say ""

# -----------------------
# Services
# -----------------------
Say "Services:"
$svcNames = @("WSearch","SysMain","wuauserv","BITS","UsoSvc","WaaSMedicSvc")
$svcState = @{}
foreach ($n in $svcNames) {
  $s = Get-Service -Name $n -ErrorAction SilentlyContinue
  if ($s) {
    $start = Get-ServiceStartType $n
    $svcState[$n] = @{ Status = $s.Status; StartType = $start }
    Say ("  {0}  Status={1}  StartType={2}" -f $n, $s.Status, $start)
  } else {
    $svcState[$n] = @{ Status = "NotFound"; StartType = "NotFound" }
    Say ("  {0}  (not found)" -f $n)
  }
}

Expect-ServiceInline $svcState "WSearch" "Disabled"
Expect-ServiceInline $svcState "SysMain" "Disabled"
Expect-ServiceInline $svcState "wuauserv" @("Disabled", "Manual")
Expect-ServiceInline $svcState "UsoSvc" "Disabled"
Expect-ServiceInline $svcState "WaaSMedicSvc" @("Disabled", "Manual")
Expect-ServiceInline $svcState "BITS" "Manual"

Say ""

# -----------------------
# Scheduled Tasks (Windows Update)
# -----------------------
Say "Scheduled Tasks:"
$taskPaths = @(
  "\Microsoft\Windows\UpdateOrchestrator\Reboot",
  "\Microsoft\Windows\UpdateOrchestrator\Schedule Scan",
  "\Microsoft\Windows\UpdateOrchestrator\USO_UxBroker",
  "\Microsoft\Windows\UpdateOrchestrator\MusUx_UpdateInterval",
  "\Microsoft\Windows\WindowsUpdate\Scheduled Start",
  "\Microsoft\Windows\WindowsUpdate\sih",
  "\Microsoft\Windows\WindowsUpdate\sihboot"
)

foreach ($tp in $taskPaths) {
  try {
    $folder = ($tp.Substring(0, $tp.LastIndexOf("\") + 1))
    $name   = ($tp.Substring($tp.LastIndexOf("\") + 1))

    $t = Get-ScheduledTask -TaskPath $folder -TaskName $name -ErrorAction Stop
    $ti = Get-ScheduledTaskInfo -TaskPath $folder -TaskName $name -ErrorAction SilentlyContinue

    $state = if ($t.State) { "$($t.State)" } else { "Unknown" }
    $enabled = $true
    try { $enabled = [bool]$t.Settings.Enabled } catch { $enabled = $true }
    if (-not $enabled) { $state = "Disabled" }

    if ($ti) { Say ("  {0}  State={1}  LastRun={2}" -f $tp, $state, $ti.LastRunTime) }
    else     { Say ("  {0}  State={1}" -f $tp, $state) }

    if ($state -ne "Disabled") {
      Add-Rec "INFO" ("Scheduled task not Disabled: {0} (State={1})" -f $tp, $state)
    }

  } catch {
    Say ("  {0}  State=NotFound" -f $tp)
  }
}
Say ""

# -----------------------
# Event Logs (max size)
# -----------------------
Say "Event Logs (max size):"
function Get-EventLogMaxMB($logName) {
  try {
    $out = & wevtutil gl $logName 2>$null
    if (-not $out) { return $null }
    $line = $out | Where-Object { $_ -match '^\s*maxSize:\s*\d+' } | Select-Object -First 1
    if (-not $line) { return $null }
    $bytes = [int64]($line -replace '^\s*maxSize:\s*','')
    return [Math]::Round($bytes / 1MB, 0)
  } catch {
    return $null
  }
}

$logMax = @{}
foreach ($ln in @("Application","System","Security")) {
  $mb = Get-EventLogMaxMB $ln
  $logMax[$ln] = $mb
  if ($mb -ne $null) { Say ("  {0,-12} MaxSizeMB={1}" -f $ln, $mb) }
  else { Say ("  {0,-12} MaxSizeMB=(unknown)" -f $ln) }
}

if ($logMax["Application"] -ne $null -and [int]$logMax["Application"] -ne $ExpectedAppLogMB) {
  Add-Rec "WARN" ("EventLog Application max is {0} MB (expected {1} MB)." -f $logMax["Application"], $ExpectedAppLogMB)
}
if ($logMax["System"] -ne $null -and [int]$logMax["System"] -ne $ExpectedSysLogMB) {
  Add-Rec "WARN" ("EventLog System max is {0} MB (expected {1} MB)." -f $logMax["System"], $ExpectedSysLogMB)
}
if ($logMax["Security"] -ne $null -and [int]$logMax["Security"] -ne $ExpectedSecLogMB) {
  Add-Rec "INFO" ("EventLog Security max is {0} MB (expected {1} MB (some builds restrict values))." -f $logMax["Security"], $ExpectedSecLogMB)
}

Say ""

# -----------------------
# NTFS LastAccess
# -----------------------
Say "NTFS LastAccess:"
try {
  $o = & fsutil behavior query disablelastaccess 2>$null
  $lastAccessVal = ($o | Select-String -Pattern '\d+' | Select-Object -First 1).Matches.Value
  if ($lastAccessVal) {
    $meaning = switch ($lastAccessVal) {
      "0" { "Enabled" }
      "1" { "Disabled (recommended for low IO)" }
      "2" { "System Managed" }
      "3" { "Enabled (user-managed)" }
      default { "Unknown" }
    }
    Say ("  DisableLastAccess={0} ({1})" -f $lastAccessVal, $meaning)

    if ($lastAccessVal -ne "1") {
      Add-Rec "WARN" ("DisableLastAccess is {0} (expected 1 = Disabled)." -f $lastAccessVal)
    }
  } else {
    Say "  DisableLastAccess=(unknown)"
    Add-Rec "WARN" "DisableLastAccess could not be determined."
  }
} catch {
  Say "  DisableLastAccess=(unable to query)"
  Add-Rec "WARN" "DisableLastAccess could not be determined."
}
Say ""

# -----------------------
# ACPI Shutdown reliability
# -----------------------
Say "ACPI Shutdown reliability:"
function Get-PowerButtonActionACDC_FromRegistry() {
  $base = "HKLM:\SYSTEM\CurrentControlSet\Control\Power\User\PowerSchemes"
  $schemeGuid = $null

  try {
    $schemeGuid = (Get-ItemProperty -Path $base -Name "ActivePowerScheme" -ErrorAction Stop).ActivePowerScheme
  } catch {
    return @{ AC=$null; DC=$null; Note="ActivePowerScheme not readable" }
  }

  if ([string]::IsNullOrWhiteSpace($schemeGuid)) {
    return @{ AC=$null; DC=$null; Note="ActivePowerScheme empty" }
  }

  $SUB_BUTTONS   = "4f971e89-eebd-4455-a8de-9e59040e7347"
  $PBUTTONACTION = "7648efa3-dd9c-4e3e-b566-50f929386280"

  $p = Join-Path $base ($schemeGuid + "\" + $SUB_BUTTONS + "\" + $PBUTTONACTION)

  $ac = $null
  $dc = $null

  try { $ac = (Get-ItemProperty -Path $p -Name "ACSettingIndex" -ErrorAction Stop).ACSettingIndex } catch {}
  try { $dc = (Get-ItemProperty -Path $p -Name "DCSettingIndex" -ErrorAction Stop).DCSettingIndex } catch {}

  return @{ AC=$ac; DC=$dc; Note=("Scheme={0}" -f $schemeGuid) }
}

$pb = Get-PowerButtonActionACDC_FromRegistry
Say ("  Power button action (AC/DC): {0}/{1} (expected 3/3 = Shutdown)" -f $pb.AC, $pb.DC)

if ($pb.AC -ne $null -and $pb.AC -ne 3) { Add-Rec "WARN" ("Power button AC action is {0} (expected 3=Shutdown)." -f $pb.AC) }
if ($pb.DC -ne $null -and $pb.DC -ne 3) { Add-Rec "WARN" ("Power button DC action is {0} (expected 3=Shutdown)." -f $pb.DC) }
if ($pb.AC -eq $null -or $pb.DC -eq $null) {
  Add-Rec "INFO" ("Power button AC/DC action could not be parsed (powercfg format/build). {0}" -f $pb.Note)
}


$wtkst = Get-RegStr "HKLM:\SYSTEM\CurrentControlSet\Control" "WaitToKillServiceTimeout"
Say ("  WaitToKillServiceTimeout:   {0} (expected 5000)" -f $wtkst)
if ($wtkst -and ($wtkst -ne "5000")) { Add-Rec "INFO" ("WaitToKillServiceTimeout is {0} (expected 5000)." -f $wtkst) }

$defAuto = Get-RegStr "Registry::HKEY_USERS\.DEFAULT\Control Panel\Desktop" "AutoEndTasks"
$defHung = Get-RegStr "Registry::HKEY_USERS\.DEFAULT\Control Panel\Desktop" "HungAppTimeout"
$defWait = Get-RegStr "Registry::HKEY_USERS\.DEFAULT\Control Panel\Desktop" "WaitToKillAppTimeout"
Say ("  HKU\.DEFAULT AutoEndTasks/Hung/Wait: {0}/{1}/{2} (expected 1/2000/5000)" -f $defAuto, $defHung, $defWait)

if ($defAuto -and $defAuto -ne "1") { Add-Rec "WARN" ("HKU\.DEFAULT AutoEndTasks is {0} (expected 1)." -f $defAuto) }
if ($defHung -and $defHung -ne "2000") { Add-Rec "INFO" ("HKU\.DEFAULT HungAppTimeout is {0} (expected 2000)." -f $defHung) }
if ($defWait -and $defWait -ne "5000") { Add-Rec "INFO" ("HKU\.DEFAULT WaitToKillAppTimeout is {0} (expected 5000)." -f $defWait) }

$cuAuto = Get-RegStr "HKCU:\Control Panel\Desktop" "AutoEndTasks"
$cuHung = Get-RegStr "HKCU:\Control Panel\Desktop" "HungAppTimeout"
$cuWait = Get-RegStr "HKCU:\Control Panel\Desktop" "WaitToKillAppTimeout"
Say ("  HKCU AutoEndTasks/Hung/Wait: {0}/{1}/{2} (expected 1/2000/5000)" -f $cuAuto, $cuHung, $cuWait)

if ($cuAuto -and $cuAuto -ne "1") { Add-Rec "WARN" ("HKCU AutoEndTasks is {0} (expected 1)." -f $cuAuto) }
if ($cuHung -and $cuHung -ne "2000") { Add-Rec "INFO" ("HKCU HungAppTimeout is {0} (expected 2000)." -f $cuHung) }
if ($cuWait -and $cuWait -ne "5000") { Add-Rec "INFO" ("HKCU WaitToKillAppTimeout is {0} (expected 5000)." -f $cuWait) }

Say ""

# -----------------------
# Reboot hint
# -----------------------
Say "Reboot hint:"
$rebootPending = $false
try {
  if (Test-Path "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Component Based Servicing\RebootPending") { $rebootPending = $true }
  if (Test-Path "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\WindowsUpdate\Auto Update\RebootRequired") { $rebootPending = $true }
  $pfr = Get-ItemProperty "HKLM:\SYSTEM\CurrentControlSet\Control\Session Manager" -Name "PendingFileRenameOperations" -ErrorAction SilentlyContinue
  if ($pfr) { $rebootPending = $true }

  if ($rebootPending) { Say "  WARN: Reboot appears pending." }
  else { Say "  OK: No reboot pending flags detected." }
} catch {
  Say "  (unable to determine reboot pending state)"
}
Say ""

# -----------------------
# CIRCLE Services
# -----------------------
Say "CIRCLE Services:"
$circleServices = @("circle-agent","circle-watchdog")
foreach ($name in $circleServices) {
  $s = $null
  try { $s = Get-CimInstance Win32_Service -Filter ("Name='{0}'" -f $name) -ErrorAction SilentlyContinue } catch {}
  if ($s) {
    Say ("  {0}  Name={1}  State={2}  StartMode={3}" -f $s.DisplayName, $s.Name, $s.State, $s.StartMode)

    if ($s.StartMode -ne "Auto") {
      Add-Rec "WARN" ("{0} StartMode is {1} (expected Auto)." -f $s.DisplayName, $s.StartMode)
    }
    if ($s.State -ne "Running") {
      Add-Rec "WARN" ("{0} is not Running (State={1})." -f $s.DisplayName, $s.State)
    }

    Validate-CircleServiceRecovery $name $s.DisplayName

  } else {
    Say ("  {0}  NOT FOUND" -f $name)
    Add-Rec "WARN" ("CIRCLE service not found: {0}" -f $name)
  }
}
Say ""

# -----------------------
# CIRCLE notify (Task Scheduler)
# -----------------------
Say "CIRCLE notify (Task Scheduler):"

$notifyExe = "C:\circle\circle-notify.exe"
$notifyTaskName = "CIRCLE circle-notify (Logon)"
$notifyProcName = "circle-notify"

if (Test-Path $notifyExe) {
  Say ("  circle-notify.exe: present ({0})" -f $notifyExe)
} else {
  Say ("  circle-notify.exe: MISSING ({0})" -f $notifyExe)
  Add-Rec "WARN" ("circle-notify.exe missing: {0}" -f $notifyExe)
}

try {
  $t = Get-ScheduledTask -TaskName $notifyTaskName -ErrorAction Stop
  $ti = Get-ScheduledTaskInfo -TaskName $notifyTaskName -ErrorAction SilentlyContinue

  $enabled = $true
  try { $enabled = [bool]$t.Settings.Enabled } catch { $enabled = $true }

  $state = if ($t.State) { "$($t.State)" } else { "Unknown" }
  if (-not $enabled) { $state = "Disabled" }

  Say ("  Task: present  Name='{0}'  State={1}" -f $notifyTaskName, $state)
  if ($ti) { Say ("  Task LastRun: {0}" -f $ti.LastRunTime) }

  if ($state -eq "Disabled") {
    Add-Rec "WARN" ("Task '{0}' is Disabled." -f $notifyTaskName)
  }

  try {
    $mi = "$($t.Settings.MultipleInstances)"
    Say ("  Task MultipleInstances: {0} (expected IgnoreNew)" -f $mi)
    if ($mi -ne "IgnoreNew") {
      Add-Rec "WARN" ("Task '{0}' MultipleInstances is {1} (expected IgnoreNew)." -f $notifyTaskName, $mi)
    }
  } catch {}

  try {
    $rc = [int]$t.Settings.RestartCount
    $ri = "$($t.Settings.RestartInterval)"
    Say ("  Task RestartCount/Interval: {0} / {1} (expected 5 / PT3M)" -f $rc, $ri)

    if ($rc -ne 5) {
      Add-Rec "WARN" ("Task '{0}' RestartCount is {1} (expected 5)." -f $notifyTaskName, $rc)
    }
    if ($ri -ne "PT3M") {
      Add-Rec "INFO" ("Task '{0}' RestartInterval is {1} (expected PT3M)." -f $notifyTaskName, $ri)
    }
  } catch {}

  try {
    $etl = "$($t.Settings.ExecutionTimeLimit)"
    Say ("  Task ExecutionTimeLimit: {0} (expected PT0S)" -f $etl)
    if ($etl -ne "PT0S") {
      Add-Rec "INFO" ("Task '{0}' ExecutionTimeLimit is {1} (expected PT0S)." -f $notifyTaskName, $etl)
    }
  } catch {}

  try {
    $a = $t.Actions | Select-Object -First 1
    if ($a -and $a.Execute) {
      Say ("  Task Action Execute: {0}" -f $a.Execute)
      if ($a.Execute -notmatch 'circle-notify\.exe$') {
        Add-Rec "WARN" ("Task '{0}' action is '{1}' (expected circle-notify.exe)." -f $notifyTaskName, $a.Execute)
      }
    }
  } catch {}

} catch {
  Say ("  Task: MISSING  Name='{0}'" -f $notifyTaskName)
  Add-Rec "WARN" ("Scheduled Task missing: {0}" -f $notifyTaskName)
}

try {
  $p = Get-Process -Name $notifyProcName -ErrorAction SilentlyContinue
  if ($p) {
    Say ("  Process: running (count={0})" -f $p.Count)
  } else {
    Say "  Process: not running (may be OK if user not logged on yet)"
  }
} catch {}

Say ""

# -----------------------
# Recommendations + summary
# -----------------------
Say "Recommendations:"
if ($recs.Count -eq 0) {
  Say "  (none) All checked settings match the expected low-IOPS profile."
} else {
  foreach ($r in $recs) { Say $r }
}
Say ""

$infoCount = ($recs | Where-Object { $_ -match '^  INFO:' }).Count
$warnCount = ($recs | Where-Object { $_ -match '^  WARN:' }).Count

if ($warnCount -eq 0 -and $infoCount -eq 0) {
  Say "TEMPLATE READY: PASS"
} elseif ($warnCount -eq 0) {
  Say ("TEMPLATE READY: PASS ({0} info)" -f $infoCount)
} else {
  Say ("TEMPLATE READY: FAIL ({0} warn, {1} info)" -f $warnCount, $infoCount)
}

Say @"

Recommended general steps for making a new template:
====================================================
1) Windows update
2) Disable Windows Update (noVirusThanks / your method)
3) Apply baseline:
  powershell -ExecutionPolicy Bypass -File .\WinVM-IOPS-Apply.ps1
4) Verify baseline:
  powershell -ExecutionPolicy Bypass -File .\WinVM-IOPS-Check.ps1
5) Clean up
  DISM /Online /Cleanup-Image /StartComponentCleanup
  cleanmgr
  compact.exe /compactOS:always
  shutdown /r /t 0
  defrag C: /X
  sdelete -z C:
  shutdown /s /t 0
6) Save as template (dashboard)
"@

Say ""
Say "Done."
