# WinVM-IOPS-Apply.ps1 (v1.5)
# Applies low-IOPS template settings on Windows 10/11
# Updates:
#  - SCM recovery policy for circle-agent + circle-watchdog (reset=86400, restart=180000 x3, failureflag=1)
#  - Replace Startup shortcut autostart with Task Scheduler for circle-notify.exe (logon, single instance, restart on failure)
#  - Simplify hive handling: assume script is run by cloud user in elevated PS; use HKCU only (plus HKU\.DEFAULT)
# RunCmd:
#  powershell -ExecutionPolicy Bypass -File .\WinVM-IOPS-Apply.ps1
# ASCII-only comments

param(
  # Pagefile size: you can pass 4096 (MB) or "4G" / "4096M"
  [string]$Pagefile = "4G",

  # Event log max sizes in MB
  [int]$AppLogMB = 4,
  [int]$SysLogMB = 4,
  [int]$SecLogMB = 20,

  [switch]$WhatIf
)

# --- Parameter validation / help ---
# If any unnamed or unknown arguments are passed, show help and exit.
if ($args.Count -gt 0) {
  Write-Host ""
  Write-Host "ERROR: Unknown parameter(s): $($args -join ' ')" -ForegroundColor Red
  Write-Host ""
  Write-Host "Usage:"
  Write-Host "  powershell -ExecutionPolicy Bypass -File WinVM-IOPS-Apply.ps1 [-Pagefile <size>] [-WhatIf]"
  Write-Host ""
  Write-Host "Options:"
  Write-Host "  -Pagefile <size>   Pagefile size (default: 4G)"
  Write-Host "                     Examples: 4096 | 4096M | 4G | 6G"
  Write-Host "  -WhatIf            Show what would be changed, do not apply"
  Write-Host ""
  exit 1
}

$ErrorActionPreference = "Inquire"

# include WU core
. "$PSScriptRoot\WindowsUpdate-Core.ps1"

function Say($msg) { Write-Host $msg }

function Require-Admin {
  $id = [Security.Principal.WindowsIdentity]::GetCurrent()
  $p  = New-Object Security.Principal.WindowsPrincipal($id)
  if (-not $p.IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)) {
    throw "Run this script as Administrator."
  }
}

function Run($desc, [ScriptBlock]$action) {
  if ($WhatIf) {
    Say ("[WHATIF] {0}" -f $desc)
  } else {
    Say ("[DO] {0}" -f $desc)
    & $action
  }
}

function Parse-SizeToMB([string]$s) {
  if ([string]::IsNullOrWhiteSpace($s)) { return 4096 }

  $t = $s.Trim().ToUpperInvariant()

  # If it's pure number, treat as MB
  if ($t -match '^\d+$') { return [int]$t }

  # Accept forms like 4G, 4096M
  if ($t -match '^(\d+)\s*G$') { return [int]$Matches[1] * 1024 }
  if ($t -match '^(\d+)\s*M$') { return [int]$Matches[1] }

  throw "Invalid -Pagefile value '$s'. Use e.g. 4096, 4096M, or 4G."
}

function Set-ServiceDisabled($name) {
  $svc = Get-Service -Name $name -ErrorAction SilentlyContinue
  if (-not $svc) { Say ("  (skip) service not found: {0}" -f $name); return }

  Run "Stop service $name" { Stop-Service -Name $name -Force -ErrorAction SilentlyContinue }
  Run "Disable service $name" { Set-Service -Name $name -StartupType Disabled -ErrorAction SilentlyContinue }
}

function Set-ServiceManual($name) {
  $svc = Get-Service -Name $name -ErrorAction SilentlyContinue
  if (-not $svc) { Say ("  (skip) service not found: {0}" -f $name); return }

  Run "Stop service $name" { Stop-Service -Name $name -Force -ErrorAction SilentlyContinue }
  Run "Set service $name to Manual" { Set-Service -Name $name -StartupType Manual -ErrorAction SilentlyContinue }
}

function Clear-EventLogBestEffort($logName) {
  Run "Clear EventLog '$logName' (best effort)" {
    & wevtutil cl $logName 2>$null | Out-Null
  }
}

function Set-EventLogMaxMB($logName, $mb) {
  # wevtutil expects bytes
  $bytes = [int64]$mb * 1024 * 1024
  Run "Set EventLog '$logName' max size to ${mb}MB" {
    & wevtutil sl $logName /ms:$bytes 2>$null | Out-Null
  }
}

function Disable-ScheduledTaskSafe($taskPath) {
  Run "Disable scheduled task $taskPath" {
    $tp = ([IO.Path]::GetDirectoryName($taskPath) + "\")
    $tn = ([IO.Path]::GetFileName($taskPath))
    Disable-ScheduledTask -TaskPath $tp -TaskName $tn -ErrorAction SilentlyContinue | Out-Null
  }
}

# Helper: set Desktop shutdown behavior values under a registry root
function Set-DesktopShutdownTweaks($root) {
  Run "Set AutoEndTasks/timeout tweaks in $root" {
    & reg add "$root\Control Panel\Desktop" /v AutoEndTasks /t REG_SZ /d 1 /f | Out-Null
    & reg add "$root\Control Panel\Desktop" /v HungAppTimeout /t REG_SZ /d 2000 /f | Out-Null
    & reg add "$root\Control Panel\Desktop" /v WaitToKillAppTimeout /t REG_SZ /d 5000 /f | Out-Null
  }
}

# Helper: set user env vars (TEMP/TMP) under a registry root
function Set-UserEnvTempTmp($root, [string]$tempPath) {
  Run "Set TEMP/TMP (User) in $root to $tempPath" {
    & reg add "$root\Environment" /v TEMP /t REG_SZ /d $tempPath /f | Out-Null
    & reg add "$root\Environment" /v TMP  /t REG_SZ /d $tempPath /f | Out-Null
  }
}

# --- SCM recovery helpers (services) ---
function Set-ServiceRecoveryPolicy($name) {
  $svc = Get-Service -Name $name -ErrorAction SilentlyContinue
  if (-not $svc) { Say ("  (skip) service not found: {0}" -f $name); return }

  Run "Set SCM recovery policy for service '$name' (reset=86400, restart=180000 x3)" {
    & sc.exe failure "$name" reset= 86400 actions= restart/180000/restart/180000/restart/180000 | Out-Null
    & sc.exe failureflag "$name" 1 | Out-Null
  }
}

# --- Task Scheduler helper (circle-notify.exe) ---
function Ensure-CircleNotifyLogonTask(
  [string]$taskName,
  [string]$exePath
) {
  if (-not (Test-Path $exePath)) {
    Say ("  WARN: circle-notify exe not found: {0} (task not created)" -f $exePath)
    return
  }

  Run "Create/Update Scheduled Task '$taskName' (At logon, single instance, restart on failure)" {
    $action  = New-ScheduledTaskAction -Execute $exePath

    # Trigger at logon for current user
    $trigger = New-ScheduledTaskTrigger -AtLogOn

    # Run only when user is logged on (no password prompt)
    $principal = New-ScheduledTaskPrincipal -UserId $env:USERNAME -LogonType Interactive -RunLevel Highest

    $settings = New-ScheduledTaskSettingsSet `
      -MultipleInstances IgnoreNew `
      -ExecutionTimeLimit (New-TimeSpan -Seconds 0) `
      -RestartCount 5 `
      -RestartInterval (New-TimeSpan -Minutes 3)

    $task = New-ScheduledTask -Action $action -Trigger $trigger -Principal $principal -Settings $settings
    Register-ScheduledTask -TaskName $taskName -InputObject $task -Force | Out-Null
  }
}

# --- Remove legacy client from Startup 
function Remove-LegacyStartupClientArtifacts {
  # Remove legacy Startup items for old client.exe autostart (best effort)
  # Kill running legacy process only if legacy startup artifact exists.
  # Robust against weird object[] values; uses Path.Combine.

  $appData = $env:APPDATA
  if ($appData -is [System.Array]) { $appData = $appData[0] }
  $appData = [string]$appData
  if ([string]::IsNullOrWhiteSpace($appData)) { return }

  $startup = [System.IO.Path]::Combine($appData, "Microsoft\Windows\Start Menu\Programs\Startup")
  if (-not (Test-Path $startup)) { return }

  $names = @(
    "client.exe",
    "client.lnk"
   )

  $candidatePaths = foreach ($n in $names) { [System.IO.Path]::Combine($startup, $n) }

  # Gather .lnk files once
  $lnkFiles = @()
  try { $lnkFiles = Get-ChildItem -Path $startup -Filter "*.lnk" -ErrorAction SilentlyContinue } catch {}

  # Detect whether any legacy startup artifact exists (file or lnk->client.exe)
  $hasLegacy = $false

  foreach ($p in $candidatePaths) {
    if (Test-Path $p) { $hasLegacy = $true; break }
  }

  if (-not $hasLegacy -and $lnkFiles.Count -gt 0) {
    foreach ($lnk in $lnkFiles) {
      try {
        $wsh = New-Object -ComObject WScript.Shell
        $sc  = $wsh.CreateShortcut($lnk.FullName)
        $tp  = $sc.TargetPath

        try { [System.Runtime.InteropServices.Marshal]::ReleaseComObject($sc) | Out-Null } catch {}
        try { [System.Runtime.InteropServices.Marshal]::ReleaseComObject($wsh) | Out-Null } catch {}

        if ($tp -and ($tp.ToLowerInvariant().EndsWith("\client.exe"))) {
          $hasLegacy = $true
          break
        }
      } catch {
        # ignore
      }
    }
  }

  if (-not $hasLegacy) {
    Say "  (skip) no legacy client.exe Startup artifact detected."
    return
  }

  # If we get here, legacy startup exists -> kill legacy client.exe first (often 2 procs due to PyInstaller)
  Run "Stop legacy client.exe processes (best effort)" {
	$legacyPaths = @(
	  "c:\circle\client.exe",
	  ($startup.ToLowerInvariant() + "\client.exe")
    )
    try {
      $procs = Get-Process -Name "client" -ErrorAction SilentlyContinue
      if ($procs) {
        foreach ($p in $procs) {
          # Prefer safe path match if we can read it
          try {
			 if ($p.Path) {
               $pp = $p.Path.ToLowerInvariant()
               if ($legacyPaths -contains $pp) {
                 Stop-Process -Id $p.Id -Force -ErrorAction SilentlyContinue
               }
            } else {
               Stop-Process -Id $p.Id -Force -ErrorAction SilentlyContinue
            } 
          } catch {
            # Fall back to killing by Id if anything weird happens
            try { Stop-Process -Id $p.Id -Force -ErrorAction SilentlyContinue } catch {}
          }
        }
      }
    } catch {}
  }

  # Now remove legacy startup items
  Run "Remove legacy Startup artifacts (client.exe / old shortcuts) from $startup" {
    foreach ($p in $candidatePaths) {
      if (Test-Path $p) {
        Remove-Item -LiteralPath $p -Force -ErrorAction SilentlyContinue
      }
    }

    # Remove any .lnk that points to *\client.exe
    foreach ($lnk in $lnkFiles) {
      try {
        $wsh = New-Object -ComObject WScript.Shell
        $sc  = $wsh.CreateShortcut($lnk.FullName)
        $tp  = $sc.TargetPath

        try { [System.Runtime.InteropServices.Marshal]::ReleaseComObject($sc) | Out-Null } catch {}
        try { [System.Runtime.InteropServices.Marshal]::ReleaseComObject($wsh) | Out-Null } catch {}

        if ($tp -and ($tp.ToLowerInvariant().EndsWith("\client.exe"))) {
          Remove-Item -LiteralPath $lnk.FullName -Force -ErrorAction SilentlyContinue
        }
      } catch {
        # ignore
      }
    }
  }
}


Require-Admin

# Cache CS once (avoid repeated CIM)
$cs = $null
try { $cs = Get-CimInstance Win32_ComputerSystem } catch {}

# Normalize pagefile MB
$pagefileMB = Parse-SizeToMB $Pagefile

Say "=== Apply low-IOPS template settings (v1.5) ==="
Say ("Computer: {0}" -f $env:COMPUTERNAME)
Say ("Pagefile target: {0} MB" -f $pagefileMB)
Say ""

# 1) Pagefile fixed size (robust: set via registry)
Run "Set fixed pagefile ${pagefileMB}/${pagefileMB} MB on C:\" {
  # Disable automatic pagefile management (CIM)
  if ($cs) {
    Set-CimInstance -InputObject $cs -Property @{ AutomaticManagedPagefile = $false } -Confirm:$false | Out-Null
  }

  # Set PagingFiles (REG_MULTI_SZ): "path initialMB maximumMB"
  $pf = "C:\pagefile.sys $pagefileMB $pagefileMB"
  & reg add "HKLM\SYSTEM\CurrentControlSet\Control\Session Manager\Memory Management" `
      /v PagingFiles /t REG_MULTI_SZ /d $pf /f | Out-Null

  # Optional: clear temp pagefile flag if present
  & reg add "HKLM\SYSTEM\CurrentControlSet\Control\Session Manager\Memory Management" `
      /v TempPageFile /t REG_DWORD /d 0 /f | Out-Null
}

# 2) Disable noisy services
Say ""
Say "Services:"
Set-ServiceDisabled "WSearch"  # Windows Search
Set-ServiceDisabled "SysMain"  # Superfetch/SysMain

# Windows Update related (lab template style)
. "$PSScriptRoot\WindowsUpdate-Core.ps1"
Set-WindowsUpdate-State -Mode Disable


# 3) Defender scheduled scan -> Never (Day=8)
Say ""
Say "Defender:"
Run "Set Defender scheduled scan day to Never (8)" {
  Set-MpPreference -ScanScheduleDay 8 -ErrorAction SilentlyContinue | Out-Null
}

# 3.1) Defender exclusions (low-IOPS friendly)
Say ""
Say "Defender Exclusions:"
$DefenderExclusions = @(
  "C:\Windows\Temp",
  "C:\Windows\SoftwareDistribution",
  "C:\ProgramData\Microsoft\Windows Defender",
  "C:\pagefile.sys"
)
foreach ($path in $DefenderExclusions) {
  Run "Add Defender exclusion $path" {
    Add-MpPreference -ExclusionPath $path -ErrorAction SilentlyContinue | Out-Null
  }
}

# 4) NTFS LastAccess disable
Say ""
Say "NTFS:"
Run "Disable NTFS LastAccess updates (DisableLastAccess=1)" {
  & fsutil behavior set disablelastaccess 1 | Out-Null
}

# 5) Event log sizes (clear then set)
Say ""
Say "Event Logs:"
Clear-EventLogBestEffort "Application"
Clear-EventLogBestEffort "System"
Clear-EventLogBestEffort "Security"

Set-EventLogMaxMB "Application" $AppLogMB
Set-EventLogMaxMB "System"      $SysLogMB
Set-EventLogMaxMB "Security"    $SecLogMB

# 6) (Optional-ish) Try to disable common Update scheduled tasks
# moved to WU core

# 7) ACPI shutdown reliability (RDP-friendly)
Say ""
Say "ACPI Shutdown reliability:"

# 7.1) Ensure Power Button action = Shutdown (AC and DC)
Run "Set power button action to Shutdown (AC/DC)" {
  & powercfg /setacvalueindex SCHEME_CURRENT SUB_BUTTONS PBUTTONACTION 3 | Out-Null
  & powercfg /setdcvalueindex SCHEME_CURRENT SUB_BUTTONS PBUTTONACTION 3 | Out-Null
  & powercfg /setactive SCHEME_CURRENT | Out-Null
}

# 7.2) Apply to the default (logon screen / disconnected situations)
Set-DesktopShutdownTweaks "HKU\.DEFAULT"

# 7.3) Apply to current user (cloud) - simplified
Say "  Apply Desktop tweaks to HKCU (current user)"
Set-DesktopShutdownTweaks "HKCU"

# 7.4) Service shutdown timeout (system-wide)
Run "Set WaitToKillServiceTimeout=5000ms" {
  & reg add "HKLM\SYSTEM\CurrentControlSet\Control" /v WaitToKillServiceTimeout /t REG_SZ /d 5000 /f | Out-Null
}

# 7.5) TEMP/TMP (User-level) for current user (cloud) - simplified
Say ""
Say "Environment variables (current user TEMP/TMP):"
$fallbackTemp = Join-Path $env:USERPROFILE "AppData\Local\Temp"
Set-UserEnvTempTmp "HKCU" $fallbackTemp

# 8) CIRCLE services (recovery hardening)
Say ""
Say "CIRCLE services (recovery hardening):"
Set-ServiceRecoveryPolicy "circle-agent"
Set-ServiceRecoveryPolicy "circle-watchdog"

# 9) CIRCLE notify autostart via Task Scheduler (logon)
Say ""
Say "CIRCLE notify autostart (Scheduled Task):"

# Remove legacy Startup autostart entries for old client.exe
Remove-LegacyStartupClientArtifacts

$notifyExe  = "C:\circle\circle-notify.exe"
$taskName   = "CIRCLE circle-notify (Logon)"
Ensure-CircleNotifyLogonTask -taskName $taskName -exePath $notifyExe

Say ""
Say "Done."
Say "NOTE: Reboot recommended (pagefile + services)."

