# WinVM-WindowsUpdate-Enable.ps1
# Re-enables Windows Update services and core scheduled tasks
# Run as Administrator
# RunCmd:
#  powershell -ExecutionPolicy Bypass -File .\WinVM-WindowsUpdate-Enable.ps1

$ErrorActionPreference = "Continue"

function Say($m) { Write-Host $m }

function Enable-ServiceSafe($name, $startType) {
  $svc = Get-Service -Name $name -ErrorAction SilentlyContinue
  if (-not $svc) {
    Say "  (skip) Service not found: $name"
    return
  }

  try {
    Say "  Set service $name StartType=$startType"
    sc.exe config $name start= $startType | Out-Null
  } catch {}

  try {
    Say "  Start service $name"
    Start-Service -Name $name -ErrorAction SilentlyContinue
  } catch {}
}

function Enable-TaskSafe($taskPath) {
  Say "  Enable scheduled task $taskPath"
  try {
    & schtasks.exe /Change /TN $taskPath /Enable | Out-Null
  } catch {}
}

Say "=== Re-enable Windows Update ==="
Say ""

# --- Services ---
Say "Services:"
Enable-ServiceSafe "wuauserv" auto
Enable-ServiceSafe "UsoSvc" auto
Enable-ServiceSafe "BITS" delayed-auto
Enable-ServiceSafe "WaaSMedicSvc" demand
Say ""

# --- Scheduled Tasks ---
Say "Scheduled Tasks:"
$tasks = @(
  "\Microsoft\Windows\UpdateOrchestrator\Schedule Scan",
  "\Microsoft\Windows\UpdateOrchestrator\USO_UxBroker",
  "\Microsoft\Windows\UpdateOrchestrator\Reboot",
  "\Microsoft\Windows\UpdateOrchestrator\MusUx_UpdateInterval",
  "\Microsoft\Windows\WindowsUpdate\Scheduled Start",
  "\Microsoft\Windows\WindowsUpdate\sih",
  "\Microsoft\Windows\WindowsUpdate\sihboot"
)

foreach ($t in $tasks) {
  Enable-TaskSafe $t
}

Say ""

# --- Optional: clear Windows Update policy keys (best effort) ---
Say "Registry policy cleanup (best effort):"
$wuPolicy = "HKLM:\SOFTWARE\Policies\Microsoft\Windows\WindowsUpdate"
if (Test-Path $wuPolicy) {
  try {
    Remove-Item -Path $wuPolicy -Recurse -Force
    Say "  Removed $wuPolicy"
  } catch {
    Say "  WARN: Could not remove $wuPolicy (may be protected)"
  }
} else {
  Say "  (none)"
}

Say ""
Say "Windows Update re-enabled."
Say "NOTE: A reboot is recommended."
