from instance.serializers import (InstanceSerializer, InstanceListItemSerializer, FlavorSerializer,
                                  LeaseSerializer)
from django.http import Http404
from django.db.models import Q
from rest_framework.viewsets import ViewSet, ModelViewSet
from rest_framework.response import Response
from rest_framework import status
from rest_framework.decorators import action
from rest_framework.permissions import IsAuthenticated

from template.serializers import InstanceFromTemplateSerializer
from instance.models import Instance, Flavor, Lease
from template.models import ImageTemplate
from template.serializers import ImageTemplateModelSerializer
from authorization.mixins import AuthorizationMixin
from guardian.shortcuts import get_perms_for_model, assign_perm


authorization = {
    "list": {"filter": ["use_instance"]},
    "create": {"model": ["instance.create_instance"]},
    "retrieve": {"object": ["use_instance"]},
    "update": {"object": ["use_instance"]},
    "destroy": {"object": ["administer_instance"]},
    "template": {"model": ["create_template_from_instance"],
                 "object": ["use_instance"]},
    "start": {"object": ["operate_instance"]},
    "stop": {"object": ["operate_instance"]},
    "suspend": {"object": ["operate_instance"]},
    "wake_up": {"object": ["operate_instance"]},
    "reset": {"object": ["operate_instance"]},
    "reboot": {"object": ["operate_instance"]},
}

update_actions = [
    "change_name",
    "change_description",
    "renew",
    "change_lease",
    "change_flavor",
    "attach_disk",
    "resize_disk",
    "add_permission",
    "remove_permission",
    "open_port",
    "close_port",
    "add_network",
    "remove_network",
    "new_password",
]


class InstanceViewSet(AuthorizationMixin, ViewSet):

    authorization = authorization
    permission_classes = [IsAuthenticated]

    def get_object(self, pk):
        try:
            return Instance.objects.get(pk=pk)
        except Instance.DoesNotExist:
            raise Http404

    def list(self, request):
        instances = self.get_objects_with_perms(request.user, "list", Instance)
        return Response(InstanceListItemSerializer(instances, many=True).data)

    def create(self, request):
        if not self.has_perms_for_model(request.user, 'create'):
            return Response({"error": "No permission to create Virtual Machine."},
                            status=status.HTTP_401_UNAUTHORIZED)
        params = request.data
        template = ImageTemplate.objects.get(Q(pk=params["template"]) &
                                             (Q(owned_by=request.user) | Q(type="SYSTEM")))
        if not template:
            return Response({"error": "No permission to use this template."},
                            status=status.HTTP_401_UNAUTHORIZED)
        flavor = Flavor.objects.get(pk=params["flavor"]) if "flavor" in params else template.flavor
        lease = Lease.objects.get(pk=params["lease"]) if "lease" in params else template.lease
        newInstance = Instance.create_instance_from_template(
            params={"name": params["name"],
                    "description": params["description"],
                    },
            flavor=flavor,
            lease=lease,
            template=template,
            networks=[{"uuid": template.network_id}],
            owner=request.user,
        )

        for perm in get_perms_for_model(Instance):
            print(perm)
            assign_perm(perm, request.user, newInstance)

        return Response(InstanceSerializer(newInstance).data, status=status.HTTP_201_CREATED)

    def retrieve(self, request, pk):
        instance = self.get_object(pk)
        if not self.has_perms_for_object(request.user, 'retrieve', instance):
            return Response({"error": "No permission to access the Virtual Machine."},
                            status=status.HTTP_401_UNAUTHORIZED)
        instance.update_status()
        return Response(instance)

    # PUT method for updating the instance own properties BUT not the related fields
    def update(self, request, pk, format=None):
        instance = self.get_object(pk)
        if not self.has_perms_for_object(request.user, 'update', instance):
            return Response({"error": "No permission to access the Virtual Machine."},
                            status=status.HTTP_401_UNAUTHORIZED)
        if "name" in request.data:
            instance.change_name(request.data["name"])
        if "description" in request.data:
            instance.change_description(request.data["description"])
        return Response(InstanceSerializer(instance).data)

    def destroy(self, request, pk, format=None):
        instance = self.get_object(pk)
        if not instance:
            return Response(status=status.HTTP_204_NO_CONTENT)
        if not self.has_perms_for_object(request.user, 'destroy', instance):
            return Response({"error": "No permission to destroy the Virtual Machine."},
                            status=status.HTTP_401_UNAUTHORIZED)
        instance.delete()
        return Response(status=status.HTTP_204_NO_CONTENT)

    @action(detail=True, methods=["post"])
    def template(self, request, pk):
        instance = self.get_object(pk)
        if not self.has_perms_for_model(request.user, 'template'):
            return Response({"error": "No permission to create template from instance."},
                            status=status.HTTP_401_UNAUTHORIZED)
        if not self.has_perms_for_object(request.user, 'template', instance):
            return Response({"error": "No permission to access the Virtual Machine."},
                            status=status.HTTP_401_UNAUTHORIZED)
        serializer = InstanceFromTemplateSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        data = serializer.validated_data
        new_template = ImageTemplate.create_from_instance(data["name"],
                                                          data["description"],
                                                          instance,
                                                          request.user)
        serializer = ImageTemplateModelSerializer(instance=new_template)
        return Response(serializer.data)

    @action(detail=True, methods=["POST"])
    def actions(self, request, pk):
        instance = self.get_object(pk)
        if not self.has_perms_for_object(request.user, request.data["action"], instance):
            return Response({"error": "No permission to use this action on the VM."},
                            status=status.HTTP_401_UNAUTHORIZED)
        success = instance.execute_common_action(action=request.data["action"])
        return Response(success)

    @action(detail=True, methods=["POST"])
    def renew_lease(self, request, pk):
        instance = self.get_object(pk)
        instance.renew()
        return Response(status=status.HTTP_200_OK)

    @action(detail=True, methods=["POST"])
    def change_lease(self, request, pk):
        instance = self.get_object(pk)
        if "lease" in request.data:
            lease = Lease.objects.get(pk=request.data["lease"])
            instance.renew(lease)
            return Response(status=status.HTTP_200_OK)
        else:
            return Response(data="No new lease provided", status=status.HTTP_400_BAD_REQUEST)

    @action(detail=True, methods=["POST"])
    def resize(self, request, pk):
        pass
        # TODO: resize not implemented in model
        # instance = self.get_object(pk)
        # if "flavor" in request.data:
        #     flavor = Flavor.objects.get(pk=request.data["flavor"])
        #     instance.resize(flavor)
        #     return Response(status=status.HTTP_200_OK)
        # else:
        #     return Response(data="No new flavor provided", status=status.HTTP_400_BAD_REQUEST)

    @action(detail=True, methods=["POST"])
    def attach_disk(self, request, pk):
        pass

    @action(detail=True, methods=["POST"])
    def detach_disk(self, request, pk):
        pass

    @action(detail=True, methods=["POST"])
    def reset_password(self, request, pk):
        instance = self.get_object(pk)
        instance.reset_password()
        return Response(status=status.HTTP_200_OK)


class FlavorViewSet(ViewSet):
    """
    Create, update or delete a flavor.
    """

    def get_object(self, pk):
        try:
            return Flavor.objects.get(pk=pk)
        except Flavor.DoesNotExist:
            raise Http404

    def list(self, request, format=None):
        flavors = Flavor.objects.all()
        return Response(FlavorSerializer(flavors, many=True).data)

    def create(self, request, format=None):
        data = request.data
        prior = 1
        if "priority" in data:
            prior = data["priority"]
        new_flavor = Flavor.create(name=data["name"],
                                   description=data["description"],
                                   ram=data["ram"],
                                   vcpu=data["vcpu"],
                                   initial_disk=data["initial_disk"],
                                   priority=prior)
        return Response(new_flavor.pk)

    def update(self, request, pk):
        return Response(status=status.HTTP_400_BAD_REQUEST)

    def partial_update(self, request, pk):
        return Response(status=status.HTTP_400_BAD_REQUEST)

    def destroy(self, request, pk):
        flavor = self.get_object(pk)
        flavor.delete()
        return Response(status=status.HTTP_204_NO_CONTENT)


class LeaseViewSet(ModelViewSet):
    queryset = Lease.objects.all()
    serializer_class = LeaseSerializer
